# MicroCat.1 へのファイルアップロードとシリアルモニタ接続スクリプト（PowerShell用）
# 使用方法: .\upload_and_monitor.ps1 [COMポート番号] [ファイル名]
# 例: .\upload_and_monitor.ps1 COM7 led_blink.py

param(
    [Parameter(Mandatory=$false)]
    [string]$Port = "",
    
    [Parameter(Mandatory=$false)]
    [string]$File = "led_blink.py",
    
    [Parameter(Mandatory=$false)]
    [int]$BaudRate = 115200
)

if ($Port -eq "") {
    Write-Host "使用方法: .\upload_and_monitor.ps1 [COMポート番号] [ファイル名]"
    Write-Host "例: .\upload_and_monitor.ps1 COM7 led_blink.py"
    Write-Host ""
    Write-Host "オプション:"
    Write-Host "  -Port        COMポート番号（例: COM7）"
    Write-Host "  -File        アップロードするファイル名（デフォルト: led_blink.py）"
    Write-Host "  -BaudRate    ボーレート（デフォルト: 115200）"
    exit 1
}

Write-Host "========================================"
Write-Host "MicroCat.1 ファイルアップロード & モニタ"
Write-Host "========================================"
Write-Host ""

# ファイルが存在するか確認
if (-not (Test-Path $File)) {
    Write-Host "エラー: ファイル '$File' が見つかりません。"
    exit 1
}

# mpremote コマンドを検出
$mpremoteCmd = $null
$mpremotePath = Get-Command mpremote -ErrorAction SilentlyContinue
if ($mpremotePath) {
    $mpremoteCmd = "mpremote"
} else {
    $pythonCmd = Get-Command python -ErrorAction SilentlyContinue
    if (-not $pythonCmd) {
        $pythonCmd = Get-Command python3 -ErrorAction SilentlyContinue
    }
    if ($pythonCmd) {
        $testResult = & $pythonCmd.Name -m mpremote --version 2>&1
        if ($LASTEXITCODE -eq 0) {
            $mpremoteCmd = "$($pythonCmd.Name) -m mpremote"
        }
    }
}

if (-not $mpremoteCmd) {
    Write-Host "エラー: mpremote が見つかりません。"
    Write-Host "インストール: pip install mpremote"
    exit 1
}

# ステップ1: ファイルアップロード
Write-Host "[1/2] ファイルをアップロード中..."
Write-Host "  ファイル: $File"
Write-Host "  COMポート: $Port"
Write-Host ""

$uploadResult = Invoke-Expression "$mpremoteCmd connect $Port cp $File `":$File`"" 2>&1

if ($LASTEXITCODE -ne 0) {
    Write-Host "アップロード失敗:"
    Write-Host $uploadResult
    exit 1
}

Write-Host "✓ アップロード成功！"
Write-Host ""

# ステップ2: シリアルモニタに接続
Write-Host "[2/2] シリアルモニタに接続中..."
Write-Host "  接続先: $Port"
Write-Host "  ボーレート: $BaudRate"
Write-Host ""
Write-Host "----------------------------------------"
Write-Host "シリアルモニタ（Ctrl+C で終了）"
Write-Host "----------------------------------------"
Write-Host ""

# mpremote の REPL モードで接続
try {
    # mpremote の repl モードを使用
    if ($mpremoteCmd -eq "mpremote") {
        & mpremote connect $Port repl
    } else {
        # python -m mpremote の場合
        $parts = $mpremoteCmd.Split(' ')
        & $parts[0] $parts[1] $parts[2] connect $Port repl
    }
} catch {
    Write-Host ""
    Write-Host "シリアルモニタ接続エラー: $_"
    Write-Host ""
    Write-Host "代替方法: 以下のコマンドで手動接続できます:"
    if ($mpremoteCmd -eq "mpremote") {
        Write-Host "  mpremote connect $Port repl"
    } else {
        Write-Host "  $mpremoteCmd connect $Port repl"
    }
    Write-Host ""
    Write-Host "または、Tera Term などのターミナルエミュレータを使用:"
    Write-Host "  ポート: $Port"
    Write-Host "  ボーレート: $BaudRate"
}

