from machine import I2C, Pin
import time
import sys
import SIM7672  # MicroCat.1 モデムライブラリ
import requests # HTTP通信ライブラリ
from adt7410 import ADT7410

# 設定ファイルから読み込み
try:
    import config
    SERVER_URL = config.SERVER_URL
    DEVICE_ID = getattr(config, 'DEVICE_ID', 'MicroCat_01')
except ImportError:
    print("エラー: config.py が見つかりません")
    print("config.example.py をコピーして config.py を作成し、設定値を入力してください")
    raise
except AttributeError as e:
    print(f"エラー: config.py に必要な設定がありません: {e}")
    raise

# --- 設定項目 ---
# 1NCE APN設定
APN = "iot.1nce.net"

# 接続設定
# PDPタイプ: 'IP' (IPv4), 'IPV6' (IPv6), 'IPV4V6' (IPv4/IPv6両対応)
PDP_TYPE = 'IP'

# 認証方式: 0=None（認証なし）, 1=PAP, 2=CHAP
# 1NCEはPAP認証方式を使用するが、ユーザー名/パスワードは不要（空欄）
# 参考: https://help.1nce.com/dev-hub/docs/data-services-apn
SECURITY_TYPE = 1  # PAP

# I2C ピン設定
I2C_SDA_PIN = 4
I2C_SCL_PIN = 5
ADT7410_ADDRESS = 0x48

# --- 初期化処理 ---

def init_modem():
    """モデムを起動し、1NCEネットワークに接続する"""
    print("モデムを初期化中...")
    try:
        m = SIM7672.modem()
        print("モデムオブジェクト作成成功")
    except Exception as e:
        raise RuntimeError(f"モデムオブジェクトの作成に失敗: {e}")
    
    try:
        print("モデムをアクティブ化中...")
        m.active(True)
        print("モデムアクティブ化成功")
    except Exception as e:
        raise RuntimeError(f"モデムのアクティブ化に失敗: {e}")
    
    # モデムが完全に起動するまで待機
    print("モデムの起動を待機中（10秒）...")
    time.sleep(10)  # より長い待機時間に変更
    
    # モデムの状態確認（可能な場合）
    print("\n--- モデム状態確認 ---")
    try:
        if hasattr(m, 'status'):
            status = m.status()
            print(f"モデムステータス: {status}")
            if status == 0:
                print("警告: モデムステータスが0です。SIMカードまたは電波状況を確認してください")
    except Exception as e:
        print(f"ステータス取得エラー: {e}")
    
    # 利用可能なメソッドを確認してデバッグ情報を出力
    print("\n利用可能なモデムメソッド:")
    modem_methods = [method for method in dir(m) if not method.startswith('_')]
    print(f"  {', '.join(modem_methods[:10])}...")  # 最初の10個だけ表示
    
    # SIMカードや信号強度の確認を試みる
    try:
        if hasattr(m, 'signal'):
            signal = m.signal()
            print(f"信号強度: {signal}")
    except:
        pass
    
    try:
        if hasattr(m, 'scan'):
            networks = m.scan()
            print(f"検出されたネットワーク数: {len(networks) if networks else 0}")
    except:
        pass
    
    print(f"\n1NCE ({APN}) に接続を試みています...")
    try:
        # connect() パラメータの説明:
        # - apn: Access Point Name（アクセスポイント名）
        #   → 1NCEのAPN: "iot.1nce.net"（必須）
        # - user, key: 認証用のユーザー名とパスワード（1NCEは空欄でOK）
        # - pdp: Packet Data Protocol タイプ（'IP', 'IPV6', 'IPV4V6'など）
        #   → データ通信に使用するプロトコルスタックを指定
        #   → 1NCEはIPv4のみサポートのため 'IP' を指定
        # - security: 認証方式（0=None, 1=PAP, 2=CHAP）
        #   → 1NCEはPAP（Password Authentication Protocol）を指定
        #   → ただし、ユーザー名/パスワードは不要（空欄）
        # 参考: https://help.1nce.com/dev-hub/docs/data-services-apn
        result = m.connect(apn=APN, user='', key='', pdp=PDP_TYPE, security=SECURITY_TYPE)
        print(f"connect() 戻り値: {result}")
        print(f"接続パラメータ: APN={APN}, PDP={PDP_TYPE}, Security={'None' if SECURITY_TYPE == 0 else ('PAP' if SECURITY_TYPE == 1 else 'CHAP')} ({SECURITY_TYPE})")
        
        # connect()の後に追加の待機時間を設ける
        print("接続確立を待機中（5秒）...")
        time.sleep(5)
    except Exception as e:
        raise RuntimeError(f"connect() 呼び出しエラー: {e}")
    
    # 接続確認（リトライロジック改善）
    retry = 0
    max_retries = 30  # 最大60秒待機（2秒×30回）
    while retry < max_retries:
        try:
            is_connected = m.isconnected()
            ip_info = m.ifconfig()
            
            # IPアドレスが0.0.0.0でない場合、接続成功とみなす
            if is_connected and ip_info[0] != '0.0.0.0':
                print(f"\nネットワーク接続成功! IP: {ip_info[0]}")
                print(f"ネットマスク: {ip_info[1]}, ゲートウェイ: {ip_info[2]}, DNS: {ip_info[3]}")
                return m
            else:
                # 接続状態とIP情報を定期的に表示
                if retry % 5 == 0:  # 5回ごとに詳細情報を表示
                    print(f"\n接続状態チェック ({retry}/{max_retries}):")
                    print(f"  isconnected(): {is_connected}")
                    print(f"  IP設定: {ip_info}")
                    
                    # モデムステータスを再確認
                    try:
                        if hasattr(m, 'status'):
                            status = m.status()
                            print(f"  モデムステータス: {status}")
                    except:
                        pass
        except Exception as e:
            print(f"isconnected() チェックエラー: {e}")
        
        retry += 1
        if retry < max_retries:
            print(f"接続待機中... ({retry}/{max_retries})")
            time.sleep(2)
    
    # 接続失敗時の詳細情報取得を試みる
    print("\n--- 接続失敗時の詳細情報 ---")
    try:
        ip_info = m.ifconfig()
        print(f"現在のIP設定: {ip_info}")
        print(f"isconnected(): {m.isconnected()}")
    except Exception as e:
        print(f"IP設定取得エラー: {e}")
    
    try:
        if hasattr(m, 'status'):
            status = m.status()
            print(f"最終モデムステータス: {status}")
    except:
        pass
    
    raise RuntimeError(f"ネットワーク接続に失敗しました（{max_retries}回リトライ後）")

def send_data_to_server(temp, modem=None):
    """サーバーへ温度データをPOST送信する"""
    payload = {
        "device_id": DEVICE_ID,
        "temperature": temp,
        "unit": "Celsius"
    }
    
    try:
        print(f"サーバーに送信中: {temp:.2f} °C...")
        # タイムアウトを30秒に設定（モバイル回線の遅延考慮）
        response = requests.post(SERVER_URL, json=payload, timeout=30)
        print(f"送信完了。ステータスコード: {response.status_code}")
        print(f"サーバー応答: {response.text}")
        response.close() # メモリ解放のため必ずclose
        return True
    except Exception as e:
        print(f"送信エラー: {type(e).__name__}: {e}")
        if modem and not modem.isconnected():
            print("  ネットワーク接続が切断されている可能性があります")
        return False

# --- メインループ ---

def main():
    # 1. I2C/センサー初期化
    i2c = I2C(0, sda=Pin(I2C_SDA_PIN), scl=Pin(I2C_SCL_PIN), freq=100000)
    sensor = ADT7410(i2c, address=ADT7410_ADDRESS)
    
    # 2. ネットワーク接続
    try:
        modem = init_modem()
    except Exception as e:
        print("=" * 50)
        print("ネットワーク接続エラー:")
        print(f"  エラー種別: {type(e).__name__}")
        print(f"  エラーメッセージ: {e}")
        print("=" * 50)
        print("\nトラブルシューティング:")
        print("1. SIMカードが正しく挿入されているか確認")
        print("2. アンテナが接続されているか確認")
        print("3. 1NCEのAPN設定が正しいか確認（iot.1nce.net）")
        print("4. 電波状況を確認（屋外や窓際で試す）")
        print("5. モデムの電源が安定しているか確認")
        return

    print("\n1分毎のデータ送信を開始します（停止は Ctrl+C）")
    print("-" * 40)

    while True:
        try:
            # ネットワーク接続状態を確認
            try:
                is_connected = modem.isconnected()
                ip_info = modem.ifconfig()
                # IPが0.0.0.0の場合は接続が切れているとみなす
                if not is_connected or ip_info[0] == '0.0.0.0':
                    print("警告: ネットワーク接続が切断されました。再接続を試みます...")
                    try:
                        # 既存の接続を切断
                        try:
                            modem.active(False)
                            time.sleep(2)
                        except:
                            pass
                        modem = init_modem()
                        print("再接続成功")
                    except Exception as e:
                        print(f"再接続失敗: {e}")
                        print("10秒後に再試行します...")
                        time.sleep(10)
                        continue
            except Exception as e:
                print(f"接続状態確認エラー: {e}")
                print("再接続を試みます...")
                try:
                    modem = init_modem()
                    print("再接続成功")
                except Exception as reconnect_error:
                    print(f"再接続失敗: {reconnect_error}")
                    print("10秒後に再試行します...")
                    time.sleep(10)
                    continue
            
            # 温度読み取り
            temp_c = sensor.read_temperature()
            print(f"温度読み取り: {temp_c:.2f} °C")
            
            # サーバーへ送信
            send_data_to_server(temp_c, modem)
            
            # 1分待機 (60秒)
            print("待機中 (60s)...")
            time.sleep(60)
            
        except KeyboardInterrupt:
            print("\nプログラムを終了します")
            break
        except Exception as e:
            print(f"ループ内エラー: {type(e).__name__}: {e}")
            sys.print_exception(e)  # 詳細なスタックトレースを表示
            print("10秒後に再試行します...")
            time.sleep(10) # エラー時は少し待って再試行

if __name__ == "__main__":
    main()