"""
ADT7410 温度センサー読み取りプログラム
MicroCat.1 用

ADT7410をI2Cで接続して温度を取得し、定期的に表示します。
"""

from machine import I2C, Pin
import time
from adt7410 import ADT7410

# I2C ピン設定（Raspberry Pi Pico 2 標準）
# I2C0: GPIO4 (SDA), GPIO5 (SCL)
I2C_SDA_PIN = 4
I2C_SCL_PIN = 5

# ADT7410 の I2C アドレス（デフォルト: 0x48）
# A0, A1 ピンの接続により変更可能: 0x48, 0x49, 0x4A, 0x4B
ADT7410_ADDRESS = 0x48

# I2C クロック周波数（100kHz または 400kHz）
I2C_FREQ = 100000  # 100kHz（標準速度）


def init_i2c():
    """
    I2Cバスを初期化
    
    Returns:
        I2C: 初期化されたI2Cオブジェクト
    """
    try:
        i2c = I2C(0, sda=Pin(I2C_SDA_PIN), scl=Pin(I2C_SCL_PIN), freq=I2C_FREQ)
        print(f"I2C初期化成功 (SDA: GPIO{I2C_SDA_PIN}, SCL: GPIO{I2C_SCL_PIN}, 周波数: {I2C_FREQ}Hz)")
        
        # 接続されているI2Cデバイスをスキャン
        devices = i2c.scan()
        if devices:
            print(f"検出されたI2Cデバイス: {[hex(addr) for addr in devices]}")
        else:
            print("警告: I2Cデバイスが見つかりません")
        
        return i2c
    except Exception as e:
        raise RuntimeError(f"I2C初期化エラー: {e}")


def read_temperature_continuous(interval=1.0, count=None):
    """
    温度を連続的に読み取り、表示する
    
    Args:
        interval: 読み取り間隔（秒）
        count: 読み取り回数（Noneの場合は無限ループ）
    """
    try:
        # I2C初期化
        i2c = init_i2c()
        
        # ADT7410初期化
        print(f"\nADT7410を初期化中... (アドレス: 0x{ADT7410_ADDRESS:02X})")
        sensor = ADT7410(i2c, address=ADT7410_ADDRESS)
        print("ADT7410初期化成功\n")
        
        # 設定レジスタの確認
        config = sensor.get_config()
        print(f"設定レジスタ: 0x{config:02X}")
        if config & 0x80:
            print("モード: 16ビット（高精度）")
        else:
            print("モード: 13ビット（標準）")
        print("-" * 50)
        
        iteration = 0
        
        while count is None or iteration < count:
            try:
                # 温度読み取り
                temp_c = sensor.read_temperature()
                temp_f = sensor.read_temperature_fahrenheit()
                
                # ステータス確認
                status = sensor.get_status()
                
                # 表示
                print(f"温度: {temp_c:6.2f}°C ({temp_f:6.2f}°F)", end="")
                
                if status['high']:
                    print(" [高温アラーム]", end="")
                if status['low']:
                    print(" [低温アラーム]", end="")
                if status['open']:
                    print(" [オープン回路]", end="")
                
                print()
                
                time.sleep(interval)
                iteration += 1
                
            except KeyboardInterrupt:
                print("\n\nプログラムを停止しました")
                break
            except Exception as e:
                print(f"\nエラー: {e}")
                time.sleep(interval)
                iteration += 1
                
    except KeyboardInterrupt:
        print("\n\nプログラムを停止しました")
    except Exception as e:
        print(f"\n初期化エラー: {e}")
        print("\nトラブルシューティング:")
        print("1. I2C接続を確認してください（SDA: GPIO4, SCL: GPIO5）")
        print("2. プルアップ抵抗（4.7kΩ～10kΩ）が接続されているか確認")
        print("3. ADT7410の電源（3.3Vまたは5V）が供給されているか確認")
        print("4. I2Cアドレスが正しいか確認（デフォルト: 0x48）")


def read_temperature_once():
    """
    温度を1回だけ読み取る
    """
    try:
        # I2C初期化
        i2c = init_i2c()
        
        # ADT7410初期化
        sensor = ADT7410(i2c, address=ADT7410_ADDRESS)
        
        # 温度読み取り
        temp_c = sensor.read_temperature()
        temp_f = sensor.read_temperature_fahrenheit()
        
        # 表示
        print(f"温度: {temp_c:.2f}°C ({temp_f:.2f}°F)")
        
        return temp_c
        
    except Exception as e:
        print(f"エラー: {e}")
        return None


if __name__ == "__main__":
    print("=" * 50)
    print("ADT7410 温度センサー読み取りプログラム")
    print("=" * 50)
    print(f"I2C設定: SDA=GPIO{I2C_SDA_PIN}, SCL=GPIO{I2C_SCL_PIN}")
    print(f"ADT7410アドレス: 0x{ADT7410_ADDRESS:02X}")
    print("停止するには Ctrl+C を押してください")
    print("=" * 50)
    
    # 連続読み取り（1秒間隔、無限ループ）
    read_temperature_continuous(interval=1.0)

