# MicroCat.1 シリアルモニタ接続スクリプト（PowerShell用）
# 使用方法: .\monitor.ps1 [COMポート番号]
# 例: .\monitor.ps1 COM7

param(
    [Parameter(Mandatory=$false)]
    [string]$Port = "",
    
    [Parameter(Mandatory=$false)]
    [int]$BaudRate = 115200
)

if ($Port -eq "") {
    Write-Host "使用方法: .\monitor.ps1 [COMポート番号]"
    Write-Host "例: .\monitor.ps1 COM7"
    Write-Host ""
    Write-Host "オプション:"
    Write-Host "  -Port        COMポート番号（例: COM7）"
    Write-Host "  -BaudRate    ボーレート（デフォルト: 115200）"
    Write-Host ""
    Write-Host "利用可能なCOMポートを検出中..."
    
    # COMポートの一覧を表示
    $ports = [System.IO.Ports.SerialPort]::GetPortNames()
    if ($ports.Count -eq 0) {
        Write-Host "利用可能なCOMポートが見つかりません。"
    } else {
        Write-Host "利用可能なCOMポート:"
        foreach ($p in $ports) {
            Write-Host "  $p"
        }
    }
    exit 1
}

Write-Host "========================================"
Write-Host "MicroCat.1 シリアルモニタ"
Write-Host "========================================"
Write-Host "接続先: $Port"
Write-Host "ボーレート: $BaudRate"
Write-Host ""
Write-Host "----------------------------------------"
Write-Host "シリアルモニタ（Ctrl+C で終了）"
Write-Host "----------------------------------------"
Write-Host ""

# mpremote コマンドを検出
$mpremoteCmd = $null
$mpremotePath = Get-Command mpremote -ErrorAction SilentlyContinue
if ($mpremotePath) {
    $mpremoteCmd = "mpremote"
} else {
    $pythonCmd = Get-Command python -ErrorAction SilentlyContinue
    if (-not $pythonCmd) {
        $pythonCmd = Get-Command python3 -ErrorAction SilentlyContinue
    }
    if ($pythonCmd) {
        $testResult = & $pythonCmd.Name -m mpremote --version 2>&1
        if ($LASTEXITCODE -eq 0) {
            $mpremoteCmd = "$($pythonCmd.Name) -m mpremote"
        }
    }
}

if (-not $mpremoteCmd) {
    Write-Host "エラー: mpremote が見つかりません。"
    Write-Host "インストール: pip install mpremote"
    Write-Host ""
    Write-Host "代替方法: PowerShellでシリアルポートを直接開く"
    Write-Host "（実装は複雑なため、mpremote の使用を推奨します）"
    exit 1
}

# mpremote の REPL モードで接続
try {
    Write-Host "接続中..."
    if ($mpremoteCmd -eq "mpremote") {
        & mpremote connect $Port repl
    } else {
        # python -m mpremote の場合
        $parts = $mpremoteCmd.Split(' ')
        & $parts[0] $parts[1] $parts[2] connect $Port repl
    }
} catch {
    Write-Host ""
    Write-Host "シリアルモニタ接続エラー: $_"
    Write-Host ""
    Write-Host "代替方法: 以下のコマンドで手動接続できます:"
    if ($mpremoteCmd -eq "mpremote") {
        Write-Host "  mpremote connect $Port repl"
    } else {
        Write-Host "  $mpremoteCmd connect $Port repl"
    }
    Write-Host ""
    Write-Host "または、Tera Term などのターミナルエミュレータを使用:"
    Write-Host "  ポート: $Port"
    Write-Host "  ボーレート: $BaudRate"
}

