"""
ADT7410 温度センサードライバー
MicroCat.1 (Raspberry Pi Pico 2互換) 用

ADT7410は16ビット高精度デジタル温度センサーです。
I2Cインターフェースで通信します。
"""

from machine import I2C, Pin
import time

# ADT7410 レジスタアドレス
ADT7410_REG_TEMP = 0x00  # 温度データレジスタ
ADT7410_REG_STATUS = 0x02  # ステータスレジスタ
ADT7410_REG_CONFIG = 0x03  # 設定レジスタ
ADT7410_REG_TEMP_HIGH = 0x04  # 高温アラーム閾値
ADT7410_REG_TEMP_LOW = 0x06  # 低温アラーム閾値

# 設定レジスタのビット定義
ADT7410_CONFIG_16BIT = 0x80  # 16ビットモード
ADT7410_CONFIG_13BIT = 0x00  # 13ビットモード（デフォルト）
ADT7410_CONFIG_FAULT_QUEUE_1 = 0x00  # フォルトキュー1回
ADT7410_CONFIG_FAULT_QUEUE_2 = 0x08  # フォルトキュー2回
ADT7410_CONFIG_FAULT_QUEUE_3 = 0x10  # フォルトキュー3回
ADT7410_CONFIG_FAULT_QUEUE_4 = 0x18  # フォルトキュー4回
ADT7410_CONFIG_CT_POLARITY = 0x04  # CTピン極性
ADT7410_CONFIG_INT_POLARITY = 0x02  # INTピン極性
ADT7410_CONFIG_INT_CT_MODE = 0x01  # INT/CTモード

# デフォルトI2Cアドレス
ADT7410_I2C_ADDR = 0x48


class ADT7410:
    """
    ADT7410 温度センサークラス
    """
    
    def __init__(self, i2c, address=ADT7410_I2C_ADDR):
        """
        ADT7410を初期化
        
        Args:
            i2c: I2Cオブジェクト（machine.I2C）
            address: I2Cアドレス（デフォルト: 0x48）
        """
        self.i2c = i2c
        self.address = address
        
        # デバイスの存在確認
        if address not in i2c.scan():
            raise ValueError(f"ADT7410が見つかりません。アドレス 0x{address:02X} を確認してください。")
        
        # 16ビットモードに設定（高精度）
        self.set_config(ADT7410_CONFIG_16BIT)
        
        # 初期化待機（最大240ms）
        time.sleep_ms(250)
    
    def set_config(self, config_value):
        """
        設定レジスタに値を書き込む
        
        Args:
            config_value: 設定値（ビットマスク）
        """
        try:
            self.i2c.writeto_mem(self.address, ADT7410_REG_CONFIG, bytes([config_value]))
        except Exception as e:
            raise RuntimeError(f"設定レジスタの書き込みに失敗しました: {e}")
    
    def get_config(self):
        """
        設定レジスタの値を読み取る
        
        Returns:
            int: 設定レジスタの値
        """
        try:
            data = self.i2c.readfrom_mem(self.address, ADT7410_REG_CONFIG, 1)
            return data[0]
        except Exception as e:
            raise RuntimeError(f"設定レジスタの読み取りに失敗しました: {e}")
    
    def read_temperature_raw(self):
        """
        温度データを生の16ビット値として読み取る
        
        Returns:
            int: 16ビットの生データ（符号付き）
        """
        try:
            # 温度レジスタから2バイト読み取り
            data = self.i2c.readfrom_mem(self.address, ADT7410_REG_TEMP, 2)
            
            # MSB（上位バイト）とLSB（下位バイト）を結合
            # MSBが最初、LSBが2番目
            raw_value = (data[0] << 8) | data[1]
            
            # 符号拡張（16ビット符号付き整数として扱う）
            if raw_value & 0x8000:
                # 負の値の場合、符号拡張
                raw_value = raw_value - 0x10000
            
            return raw_value
        except Exception as e:
            raise RuntimeError(f"温度データの読み取りに失敗しました: {e}")
    
    def read_temperature(self):
        """
        温度を摂氏（℃）で読み取る
        
        Returns:
            float: 温度（℃）
        """
        raw_value = self.read_temperature_raw()
        
        # 16ビットモードの場合、分解能は0.0078℃
        # raw_valueは128倍された値なので、128で割る
        temperature = raw_value / 128.0
        
        return temperature
    
    def read_temperature_fahrenheit(self):
        """
        温度を華氏（°F）で読み取る
        
        Returns:
            float: 温度（°F）
        """
        celsius = self.read_temperature()
        fahrenheit = celsius * 9.0 / 5.0 + 32.0
        return fahrenheit
    
    def get_status(self):
        """
        ステータスレジスタを読み取る
        
        Returns:
            dict: ステータス情報
        """
        try:
            status = self.i2c.readfrom_mem(self.address, ADT7410_REG_STATUS, 1)[0]
            return {
                'rt': bool(status & 0x80),  # 温度準備完了
                'open': bool(status & 0x40),  # オープン回路検出
                'high': bool(status & 0x20),  # 高温アラーム
                'low': bool(status & 0x10),  # 低温アラーム
            }
        except Exception as e:
            raise RuntimeError(f"ステータスレジスタの読み取りに失敗しました: {e}")

