# MicroCat.1 LED点滅プログラム

MicroCat.1 用の MicroPython プログラムです。2つのLED（LED1: GPIO25、LED2: GPIO29）を交互に点滅させます。

## ハードウェア情報

- **LED1**: GPIO25に接続されたユーザーLED（緑色）
- **LED2**: GPIO29に接続されたインジケータLED（青色）
- **互換性**: Raspberry Pi Pico 2 と互換

## 必要なもの

- MicroCat.1 基板
- USB Type-C ケーブル
- PC（Windows/Mac/Linux）
- ターミナルエミュレータ（Tera Term、PuTTY、minicom など）

## セットアップ手順

### 1. ファームウェアの確認

MicroCat.1 には既に MicroPython ファームウェアが書き込まれています。
もし最新のファームウェアが必要な場合は、以下からダウンロードしてください：
- https://github.com/mechatrax/micropython/releases

### 2. USB接続

1. MicroCat.1 を USB Type-C ケーブルで PC に接続します
2. PC がシリアルポートとして認識するのを待ちます
   - Windows: デバイスマネージャーで COM ポート番号を確認
   - Mac/Linux: `/dev/ttyUSB0` や `/dev/ttyACM0` などのデバイスファイルが作成されます

### 3. ターミナルエミュレータで接続

#### Windows の場合（Tera Term を使用）

1. Tera Term を起動
2. 「シリアルポート」を選択
3. COM ポート番号を選択（例: COM3）
4. ボーレート: 115200 を設定
5. 「OK」をクリック

#### Mac/Linux の場合（screen を使用）

```bash
screen /dev/ttyUSB0 115200
```

または minicom を使用：

```bash
minicom -D /dev/ttyUSB0 -b 115200
```

### 4. MicroPython の動作確認

ターミナルに接続すると、MicroPython のプロンプト（`>>>`）が表示されます。
以下のコマンドで動作確認できます：

```python
>>> import machine
>>> machine.freq()
```

## Cursorから直接アップロードする方法

**重要**: `pip install thonny` ではCursorから直接アップロードできません。ThonnyはGUIアプリケーションのため、コマンドラインツールが必要です。

### 推奨方法: mpremote を使用

```bash
# 1. mpremote をインストール
pip install mpremote

# 2. ファイルをアップロード（自動検出）
mpremote cp led_blink.py :led_blink.py

# 3. main.py として保存して自動実行
mpremote cp led_blink.py :main.py
```

これで、Cursorのターミナルから直接MicroCat.1にプログラムをアップロードできます。

## プログラムの実行方法

### 方法1: ターミナルから直接実行

1. ターミナルエミュレータで MicroPython プロンプト（`>>>`）が表示されていることを確認
2. 以下のコマンドを入力して実行：

```python
>>> from machine import Pin
>>> import time
>>> led1 = Pin(25, Pin.OUT)
>>> led2 = Pin(29, Pin.OUT)
>>> while True:
...     led1.on()
...     led2.off()
...     time.sleep(0.5)
...     led1.off()
...     led2.on()
...     time.sleep(0.5)
```

### 方法2: ファイルをアップロードして実行（推奨）

#### 2-1. ファイルのアップロード

**Cursor/コマンドラインから直接アップロード（推奨）**

**方法A: mpremote を使用（MicroPython公式ツール、推奨）**

```bash
# mpremote をインストール
pip install mpremote

# ファイルをアップロード（自動検出、最も簡単）
mpremote cp led_blink.py :led_blink.py

# ポートを指定する場合
# Windows の場合
mpremote connect COM3 cp led_blink.py :led_blink.py

# Mac/Linux の場合
mpremote connect /dev/ttyUSB0 cp led_blink.py :led_blink.py

# main.py として保存して自動実行
mpremote cp led_blink.py :main.py

# プログラムを直接実行（アップロードせずに）
mpremote run led_blink.py
```

**便利なスクリプトを使用する場合**

プロジェクトに含まれているアップロードスクリプトを使用できます：

```powershell
# アップロードのみ（PowerShell）
.\upload.ps1 COM7 led_blink.py

# アップロード + シリアルモニタ接続（推奨）
.\upload_and_monitor.ps1 COM7 led_blink.py

# シリアルモニタのみ接続
.\monitor.ps1 COM7

# コマンドプロンプト（cmd.exe）の場合
upload.bat COM7 led_blink.py

# Mac/Linux の場合（実行権限を付与）
chmod +x upload.sh
./upload.sh /dev/ttyUSB0 led_blink.py
```

**注意**: 
- PowerShellでは、現在のディレクトリにあるスクリプトを実行する際に`.\`を付ける必要があります
- PowerShellスクリプト（`.ps1`）の実行が許可されていない場合は、実行ポリシーを変更してください：
  ```powershell
  Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser
  ```

**シリアルモニタの使い方**

`upload_and_monitor.ps1` または `monitor.ps1` を実行すると、MicroPythonのREPL（対話型シェル）に接続されます：

- プログラムを実行: `exec(open('led_blink.py').read())`
- プログラムを停止: `Ctrl+C`
- シリアルモニタを終了: `Ctrl+C`（REPLから抜ける）

**方法B: ampy を使用**

```bash
# ampy をインストール
pip install adafruit-ampy

# ファイルをアップロード
# Windows の場合
ampy --port COM3 put led_blink.py

# Mac/Linux の場合
ampy --port /dev/ttyUSB0 put led_blink.py

# main.py として保存
ampy --port COM3 put led_blink.py main.py
```

**注意**: ThonnyはGUIアプリケーションのため、`pip install thonny`ではCursorから直接アップロードできません。上記のコマンドラインツール（mpremote または ampy）を使用してください。

**GUIツールを使用する場合（Thonny）**

1. Thonny IDE を公式サイトからインストール（https://thonny.org/）
   - **注意**: `pip install thonny` では完全なIDEはインストールされません
2. Thonny を起動
3. 「実行」→「インタープリターを構成」で MicroPython (Raspberry Pi Pico) を選択
4. ポートを選択（MicroCat.1 の COM ポート）
5. `led_blink.py` を開く
6. 「ファイル」→「名前を付けて保存」→「MicroPython デバイス」を選択
7. ファイル名を `led_blink.py` として保存

#### 2-2. プログラムの実行

ターミナルエミュレータで：

```python
>>> exec(open('led_blink.py').read())
```

または、`main.py` として保存すると、起動時に自動実行されます：

```python
>>> import os
>>> os.rename('led_blink.py', 'main.py')
```

その後、リセットボタン（SW1）を押すか、電源を再投入すると自動的に実行されます。

### 方法3: main.py として自動実行

1. `led_blink.py` を `main.py` にリネーム
2. MicroCat.1 にアップロード
3. リセットボタン（SW1）を押すか、電源を再投入
4. 自動的に LED が点滅を開始します

## プログラムの停止方法

- **ターミナルから実行している場合**: `Ctrl+C` を押す
- **自動実行している場合**: リセットボタン（SW1）を押すか、電源を切断

## カスタマイズ

`led_blink.py` の `blink_leds()` 関数のパラメータを変更することで、動作をカスタマイズできます：

```python
# 点滅間隔を1秒に変更
blink_leds(interval=1.0)

# 10回だけ点滅して停止
blink_leds(interval=0.5, count=10)
```

## トラブルシューティング

### LED が点灯しない

1. GPIO ピン番号が正しいか確認（LED1: GPIO25、LED2: GPIO29）
2. プログラムが正しく実行されているか確認
3. ハードウェアの接続を確認

### シリアルポートが認識されない

1. USB ケーブルが正しく接続されているか確認
2. デバイスマネージャー（Windows）や `lsusb`（Linux）でデバイスを確認
3. ドライバが正しくインストールされているか確認

### プログラムが実行されない

1. MicroPython のプロンプト（`>>>`）が表示されているか確認
2. ファイルが正しくアップロードされているか確認
3. シンタックスエラーがないか確認（ターミナルでエラーメッセージを確認）

## その他のプログラム

### ADT7410 温度センサー

ADT7410温度センサーの接続と使用方法については、[ADT7410_SETUP.md](ADT7410_SETUP.md) を参照してください。

### BME280 環境センサー

BME280環境センサー（温度・湿度・気圧）の接続と使用方法については、[BME280_SETUP.md](BME280_SETUP.md) を参照してください。

## 参考資料

- MicroCat.1 Wiki: https://github.com/mechatrax/microcat1/wiki
- MicroPython ドキュメント: https://docs.micropython.org/
- Raspberry Pi Pico 2 ドキュメント: https://www.raspberrypi.com/documentation/microcontrollers/

## ライセンス

このプログラムは自由に使用・改変できます。

