# MicroCat.1 へのファイルアップロードスクリプト（PowerShell用）
# 使用方法: .\upload.ps1 [COMポート番号] [ファイル名]
# 例: .\upload.ps1 COM7 led_blink.py

param(
    [Parameter(Mandatory=$false)]
    [string]$Port = "",
    
    [Parameter(Mandatory=$false)]
    [string]$File = "led_blink.py"
)

if ($Port -eq "") {
    Write-Host "使用方法: .\upload.ps1 [COMポート番号] [ファイル名]"
    Write-Host "例: .\upload.ps1 COM7 led_blink.py"
    Write-Host ""
    Write-Host "COMポート番号を自動検出する場合は、mpremote を直接使用してください:"
    Write-Host "  mpremote cp led_blink.py :led_blink.py"
    exit 1
}

Write-Host "MicroCat.1 に $File をアップロード中..."
Write-Host "COMポート: $Port"

# ファイルが存在するか確認
if (-not (Test-Path $File)) {
    Write-Host "エラー: ファイル '$File' が見つかりません。"
    exit 1
}

# mpremote コマンドを検出（複数の方法を試す）
$mpremoteCmd = $null

# 方法1: 直接 mpremote コマンドを試す
$mpremotePath = Get-Command mpremote -ErrorAction SilentlyContinue
if ($mpremotePath) {
    $mpremoteCmd = "mpremote"
} else {
    # 方法2: python -m mpremote を試す
    $pythonCmd = Get-Command python -ErrorAction SilentlyContinue
    if (-not $pythonCmd) {
        $pythonCmd = Get-Command python3 -ErrorAction SilentlyContinue
    }
    if ($pythonCmd) {
        # python -m mpremote が利用可能か確認
        $testResult = & $pythonCmd.Name -m mpremote --version 2>&1
        if ($LASTEXITCODE -eq 0) {
            $mpremoteCmd = "$($pythonCmd.Name) -m mpremote"
        }
    }
}

if (-not $mpremoteCmd) {
    Write-Host "mpremote が見つかりません。"
    Write-Host "インストール: pip install mpremote"
    Write-Host ""
    Write-Host "インストール後、PowerShellを再起動するか、以下のコマンドでパスを更新してください:"
    Write-Host "  `$env:Path = [System.Environment]::GetEnvironmentVariable('Path','Machine') + ';' + [System.Environment]::GetEnvironmentVariable('Path','User')"
    exit 1
}

# ファイル名を抽出（パスからファイル名だけを取得）
$fileName = Split-Path -Leaf $File

# アップロード実行
Write-Host "mpremote コマンド: $mpremoteCmd"
Write-Host "アップロード先ファイル名: $fileName"
$result = Invoke-Expression "$mpremoteCmd connect $Port cp $File `":$fileName`"" 2>&1

if ($LASTEXITCODE -eq 0) {
    Write-Host ""
    Write-Host "アップロード成功！"
    Write-Host "ファイルを実行するには、ターミナルで以下を実行:"
    Write-Host "  exec(open('$fileName').read())"
} else {
    Write-Host ""
    Write-Host "アップロード失敗。COMポート番号とファイル名を確認してください。"
    Write-Host $result
    exit 1
}

