from machine import I2C, Pin, SPI
import time
import sys
import framebuf
import math
import SIM7672  # MicroCat.1 モデムライブラリ
import requests # HTTP通信ライブラリ
from adt7410 import ADT7410

# 電子ペーパー用ドライバ
try:
    from epd2in66b import EPD_2in9_B
    EPD_AVAILABLE = True
except ImportError:
    print("警告: epd2in66b.py が見つかりません。")
    EPD_AVAILABLE = False

# 設定ファイル
try:
    import config
    SERVER_URL = config.SERVER_URL
    DEVICE_ID = getattr(config, 'DEVICE_ID', 'MicroCat_01')
except Exception as e:
    print(f"設定読み込みエラー: {e}")
    raise

# --- 設定項目 ---
APN = "iot.1nce.net"
PDP_TYPE = 'IP'
SECURITY_TYPE = 1
LAT = "35.4093"
LON = "134.2507"
# Open-Meteo API URL (現在時刻から予報を取得、降水確率も含む)
WEATHER_URL = f"https://api.open-meteo.com/v1/forecast?latitude={LAT}&longitude={LON}&hourly=temperature_2m,weathercode,precipitation_probability&forecast_days=1"

# I2Cピン（ADT7410）
I2C_SDA_PIN = 4
I2C_SCL_PIN = 5
ADT7410_ADDRESS = 0x48

# 送信間隔（秒）: 5分
SEND_INTERVAL = 300

# 天気コード変換マップ
WEATHER_CODES = {
    0: "Clear", 1: "Mainly Clear", 2: "Partly Cloudy", 3: "Overcast",
    45: "Fog", 48: "Fog", 51: "Drizzle", 53: "Drizzle", 55: "Drizzle",
    61: "Slight Rain", 63: "Rain", 65: "Heavy Rain", 71: "Snow", 73: "Snow",
    75: "Heavy Snow", 80: "Showers", 81: "Showers", 82: "Showers",
    95: "Thunderstorm"
}

# --- 関数定義 ---

def init_modem():
    """モデムを起動し、LTEに接続する"""
    print("モデム接続中...")
    try:
        # パラメータの確認
        print(f"APN: {APN}, PDP_TYPE: {PDP_TYPE}, SECURITY_TYPE: {SECURITY_TYPE}")
        if APN is None or PDP_TYPE is None or SECURITY_TYPE is None:
            raise ValueError("接続パラメータがNoneです")
        
        print("モデムオブジェクト作成中...")
        m = SIM7672.modem()
        print("モデムオブジェクト作成完了")
        
        print("モデムをアクティブ化中...")
        m.active(True)
        print("モデムアクティブ化完了、10秒待機...")
        time.sleep(10)
        print("待機完了")
        
        # 接続前にモデムの状態を確認
        try:
            print("接続前の状態確認...")
            if hasattr(m, 'status'):
                status = m.status()
                print(f"モデムステータス: {status}")
                if status == 0:
                    print("警告: モデムステータスが0です。SIMカードまたは電波状況を確認してください")
            if hasattr(m, 'signal'):
                signal = m.signal()
                print(f"信号強度: {signal}")
                if signal is not None and signal < 10:
                    print("警告: 信号強度が弱い可能性があります")
        except Exception as e:
            print(f"状態確認エラー（無視）: {e}")
        
        # 既に接続されているか確認
        try:
            is_already_connected = m.isconnected()
            print(f"既存接続状態: {is_already_connected}")
            if is_already_connected:
                print("既に接続されています")
                return m
        except Exception as e:
            print(f"接続状態確認エラー: {e}")
        
        # connect()の呼び出しをtry-exceptで囲む
        print(f"接続開始: APN={APN}, PDP={PDP_TYPE}, Security={SECURITY_TYPE}")
        print("注意: connect()は最大60秒待機します...")
        try:
            # 1NCEはPAP認証だが、ユーザー名/パスワードは不要（空欄）
            # connect()は内部でタイムアウトを持っている可能性があるが、
            # ブロッキングする場合は手動で中断する必要がある
            m.connect(apn=APN, user='', key='', pdp=PDP_TYPE, security=SECURITY_TYPE)
            print("connect()呼び出し完了")
        except KeyboardInterrupt:
            print("connect()が中断されました")
            raise
        except Exception as e:
            print(f"connect()エラー: {type(e).__name__}: {e}")
            # connect()が失敗しても、既に接続されている可能性があるので続行
            print("connect()エラーを無視して接続確認を続行します...")
        
        print("接続確認ループ開始...")
        retry = 0
        while retry < 30:
            try:
                is_connected = m.isconnected()
                ip_info = m.ifconfig()
                
                # 詳細情報を表示（5回ごと、または接続状態が変化した時）
                if retry % 5 == 0 or retry == 0:
                    print(f"接続確認 ({retry+1}/30):")
                    print(f"  isconnected()={is_connected}")
                    print(f"  IP設定: {ip_info}")
                    if hasattr(m, 'status'):
                        try:
                            status = m.status()
                            print(f"  モデムステータス: {status}")
                        except:
                            pass
                    if hasattr(m, 'signal'):
                        try:
                            signal = m.signal()
                            print(f"  信号強度: {signal}")
                        except:
                            pass
                else:
                    print(f"接続確認 ({retry+1}/30): isconnected()={is_connected}")
                
                # IPアドレスが0.0.0.0でない場合、接続成功とみなす
                if is_connected is True and ip_info[0] != '0.0.0.0':
                    print(f"LTE接続成功! IP: {ip_info[0]}")
                    return m
            except Exception as e:
                print(f"isconnected()チェックエラー: {e}")
            time.sleep(2)
            retry += 1
        
        # 接続失敗時の詳細情報
        print("\n--- 接続失敗時の詳細情報 ---")
        try:
            ip_info = m.ifconfig()
            print(f"最終IP設定: {ip_info}")
            print(f"最終isconnected(): {m.isconnected()}")
        except Exception as e:
            print(f"最終状態取得エラー: {e}")
        
        raise RuntimeError("ネットワーク接続失敗（30回リトライ後）")
    except Exception as e:
        print(f"init_modem()エラー: {type(e).__name__}: {e}")
        import sys
        sys.print_exception(e)
        raise

def get_weather_forecast():
    """Open-Meteoから天気予報を取得し、降水確率を10%単位に補正する"""
    print("天気予報を取得中...")
    try:
        res = requests.get(WEATHER_URL, timeout=30)
        if res.status_code == 200:
            data = res.json()
            res.close()
            # hourlyデータから直近6件を抽出
            hourly = data.get('hourly', {})
            temps = hourly.get('temperature_2m', [])
            codes = hourly.get('weathercode', [])
            precips = hourly.get('precipitation_probability', [])
            
            # データの存在確認
            if not temps or not codes:
                print("警告: 天気データが空です")
                return None
            
            # 6時間分のデータをリスト化
            forecast = []
            for i in range(6):
                if i < len(temps) and temps[i] is not None:
                    code = codes[i] if (i < len(codes) and codes[i] is not None) else 0
                    
                    # 降水確率を取得し、10%単位に四捨五入
                    raw_precip = precips[i] if (i < len(precips) and precips[i] is not None) else 0
                    precip_10 = int(round(raw_precip / 10) * 10)  # 10%単位に四捨五入
                    
                    weather_text = WEATHER_CODES.get(code, f"Code:{code}")
                    forecast.append({
                        "temp": float(temps[i]),
                        "desc": str(weather_text),
                        "code": int(code),
                        "precip": precip_10  # 補正後の値を格納
                    })
            return forecast if forecast else None
        else:
            print(f"APIエラー: ステータスコード {res.status_code}")
            res.close()
    except Exception as e:
        print(f"天気取得エラー: {e}")
    return None

def draw_weather_icon(fb_black, fb_red, x, y, code):
    """
    天気コードに応じたアイコンを描画する
    x, y: アイコンの左上座標
    fb_black: 黒バッファ（framebufオブジェクト）
    fb_red: 赤バッファ（framebufオブジェクト）
    code: 天気コード
    """
    # 背景が白(反転仕様)のため、0xffで描画すると黒、0x00で描画すると白になる
    # fb_blackへの0xffは黒、fb_redへの0x00は赤
    
    # 晴れ (☀️) - コード 0, 1
    if code in [0, 1]:
        # 赤い太陽（中心の四角 + 光線）
        fb_red.fill_rect(x + 10, y + 10, 4, 4, 0x00)  # 中心
        # 光線を8方向に描画
        fb_red.line(x + 12, y + 8, x + 12, y + 6, 0x00)  # 上
        fb_red.line(x + 12, y + 14, x + 12, y + 16, 0x00)  # 下
        fb_red.line(x + 8, y + 12, x + 6, y + 12, 0x00)  # 左
        fb_red.line(x + 14, y + 12, x + 16, y + 12, 0x00)  # 右
        fb_red.line(x + 10, y + 10, x + 8, y + 8, 0x00)  # 左上
        fb_red.line(x + 14, y + 10, x + 16, y + 8, 0x00)  # 右上
        fb_red.line(x + 10, y + 14, x + 8, y + 16, 0x00)  # 左下
        fb_red.line(x + 14, y + 14, x + 16, y + 16, 0x00)  # 右下
    
    # 曇り (☁️) - コード 2, 3, 45, 48
    elif code in [2, 3, 45, 48]:
        # 黒い雲（複数の四角を組み合わせ）
        fb_black.fill_rect(x + 4, y + 12, 6, 4, 0xff)
        fb_black.fill_rect(x + 8, y + 10, 8, 6, 0xff)
        fb_black.fill_rect(x + 14, y + 12, 6, 4, 0xff)
    
    # 雨 (☔) - コード 51, 53, 55, 61, 63, 65, 80, 81, 82
    elif code in [51, 53, 55, 61, 63, 65, 80, 81, 82]:
        # 雲（黒）
        fb_black.fill_rect(x + 4, y + 8, 5, 3, 0xff)
        fb_black.fill_rect(x + 8, y + 6, 8, 5, 0xff)
        fb_black.fill_rect(x + 14, y + 8, 5, 3, 0xff)
        # 雨粒（黒の縦線）
        fb_black.vline(x + 8, y + 12, 6, 0xff)
        fb_black.vline(x + 14, y + 13, 5, 0xff)
        fb_black.vline(x + 20, y + 12, 6, 0xff)
    
    # 雪 (☃️) - コード 71, 73, 75
    elif code in [71, 73, 75]:
        # 雪だるま風（黒）
        fb_black.fill_rect(x + 10, y + 6, 4, 4, 0xff)  # 頭
        fb_black.fill_rect(x + 8, y + 12, 8, 6, 0xff)  # 体
        # 雪の結晶（簡易版 - 十字）
        fb_black.hline(x + 12, y + 4, 1, 0xff)
        fb_black.hline(x + 12, y + 20, 1, 0xff)
        fb_black.vline(x + 8, y + 12, 1, 0xff)
        fb_black.vline(x + 16, y + 12, 1, 0xff)
    
    # 雷 (⚡) - コード 95
    elif code == 95:
        # 雲（黒）
        fb_black.fill_rect(x + 4, y + 8, 5, 3, 0xff)
        fb_black.fill_rect(x + 8, y + 6, 8, 5, 0xff)
        # 稲妻（ジグザグ線）
        fb_black.line(x + 14, y + 12, x + 18, y + 16, 0xff)
        fb_black.line(x + 18, y + 16, x + 16, y + 18, 0xff)
        fb_black.line(x + 16, y + 18, x + 20, y + 20, 0xff)
    
    # その他（不明なコード）
    else:
        # 簡易的な雲のアイコン
        fb_black.fill_rect(x + 8, y + 10, 8, 4, 0xff)

def update_epaper_with_weather(sensor_temp, forecast):
    """電子ペーパーにセンサー温度と天気予報を表示（背景白）"""
    if not EPD_AVAILABLE:
        return
    try:
        epd = EPD_2in9_B()
        # 背景を白に初期化
        epd.imageblack.fill(0x00)  # 0x00 -> 反転して 0xFF (白) が送信される
        epd.imagered.fill(0xff)    # 0xff -> 反転して 0x00 (白/色なし) が送信される
        
        # ヘッダー（黒）
        epd.imageblack.rect(0, 0, 152, 35, 0xff)  # 外枠
        epd.imageblack.text("YAZU WEATHER", 25, 12, 0xff)  # 黒文字
        
        # 現在のセンサー温度（赤）
        epd.imagered.text("SENSOR:", 10, 45, 0x00)
        # sensor_tempがNoneでないことを確認
        if sensor_temp is not None:
            temp_str = f"{float(sensor_temp):.1f} C"
        else:
            temp_str = "N/A"
        epd.imagered.text(temp_str, 75, 45, 0x00)
        
        # 6時間予報（アイコン表示）
        y_pos = 70
        if forecast and isinstance(forecast, list):
            for i, item in enumerate(forecast[:6]):
                if not isinstance(item, dict):
                    continue
                # 時間ラベル (+1h等)
                time_label = f"+{i+1}h"
                epd.imageblack.text(time_label, 5, y_pos + 8, 0xff)
                
                # 天気アイコンの描画
                weather_code = item.get('code', 0)
                draw_weather_icon(epd.imageblack, epd.imagered, 35, y_pos, weather_code)
                
                # 予想気温（赤）
                temp = item.get('temp')
                if temp is not None:
                    temp_str = f"{float(temp):.0f}C"
                else:
                    temp_str = "N/A"
                epd.imagered.text(temp_str, 85, y_pos + 8, 0x00)
                
                # 降水確率（黒）
                precip = item.get('precip', 0)
                if precip is not None:
                    # "/" を挟んで表示: "25C / 30%"
                    epd.imageblack.text("/", 110, y_pos + 8, 0xff)
                    precip_str = f"{int(precip)}%"
                    epd.imageblack.text(precip_str, 120, y_pos + 8, 0xff)
                
                y_pos += 35  # アイコン用に間隔を広げる
                if y_pos > 260:  # 画面の高さを超えないように
                    break
        else:
            epd.imageblack.text("Failed to fetch", 10, 110, 0xff)
        
        print("e-ink更新中...")
        epd.display()
        epd.sleep()
    except Exception as e:
        print(f"描画エラー: {e}")
        import sys
        sys.print_exception(e)

def main():
    # 1. センサー初期化
    i2c = I2C(0, sda=Pin(I2C_SDA_PIN), scl=Pin(I2C_SCL_PIN))
    sensor = ADT7410(i2c, address=ADT7410_ADDRESS)
    
    # 2. ネットワーク接続
    try:
        modem = init_modem()
    except Exception as e:
        print(f"初期化失敗: {type(e).__name__}: {e}")
        import sys
        sys.print_exception(e)
        return

    while True:
        try:
            # センサー温度取得
            current_temp = sensor.read_temperature()
            if current_temp is not None:
                print(f"測定: {current_temp:.2f} C")
            else:
                print("警告: 温度取得失敗")
                current_temp = 0.0
            
            # 天気予報取得
            forecast_data = get_weather_forecast()
            
            # 表示更新
            update_epaper_with_weather(current_temp, forecast_data)
            
            # サーバーへセンサーデータを送信
            if current_temp is not None:
                payload = {"device_id": DEVICE_ID, "temperature": current_temp}
                try:
                    res = requests.post(SERVER_URL, json=payload, timeout=30)
                    print(f"送信成功: {res.status_code}")
                    res.close()
                except Exception as e:
                    print(f"送信失敗: {e}")
            
            print(f"完了。{SEND_INTERVAL}秒待機...")
            time.sleep(SEND_INTERVAL)
            
        except KeyboardInterrupt:
            print("終了します")
            break
        except Exception as e:
            print(f"ループエラー: {e}")
            import sys
            sys.print_exception(e)
            time.sleep(10)

if __name__ == "__main__":
    main()